/**
 * SuiteQL Query Tool - Sample Plugin: Query Logger
 *
 * This plugin demonstrates the plugin architecture capabilities:
 * - Server-side hooks (onAfterQuery - logs queries to console)
 * - Client-side hooks (onInit, onAfterQuery, onEditorChange)
 * - UI injection (status bar indicator)
 * - Settings schema (enable/disable logging)
 *
 * To install this plugin:
 * 1. Upload this file to your NetSuite File Cabinet
 * 2. Set CONFIG.PLUGIN_FOLDER_ID to the folder ID containing this file
 * 3. Refresh the SuiteQL Query Tool
 *
 * Plugin file naming convention: *.sqt-plugin.js or *.sqt-plugin.json
 */
(function() {
    return {
        // Required: Unique plugin name
        name: 'query-logger',

        // Required: Plugin version
        version: '1.0.0',

        // Optional: Minimum SQT version required
        minAppVersion: '2026.1',

        // Optional: Plugin description
        description: 'Logs query execution details to the browser console and displays a status indicator.',

        // Optional: Author information
        author: 'SuiteQL Query Tool',

        // Optional: Dependencies on other plugins (loaded first)
        dependencies: [],

        // Optional: Built-in features to disable
        // Available features: 'ai', 'ai-chat', 'ai-explain', 'ai-validate', 'ai-nlbar',
        // 'export', 'export-airtable', 'export-google-sheets', 'local-library',
        // 'remote-library', 'workbooks', 'tables-reference', 'schema-explorer',
        // 'doc-generator', 'share', 'history', 'dark-mode', 'focus-mode', 'format', 'options'
        disables: [],

        // Optional: Server-side functionality
        server: {
            // Server-side hooks
            hooks: {
                // Called after query execution completes successfully
                onAfterQuery: function(data, plugin) {
                    // Log query details to NetSuite execution log
                    // Note: 'modules' is available in server context
                    log.audit({
                        title: 'Query Logger',
                        details: JSON.stringify({
                            rowCount: data.response.rowCount,
                            elapsedTime: data.response.elapsedTime,
                            queryPreview: data.query.substring(0, 100)
                        })
                    });

                    // Return data unchanged (or modify if needed)
                    return data;
                }
            },

            // Custom server-side handlers (accessible via fetch)
            // Handler name becomes: plugin_query-logger_getStats
            handlers: {
                getStats: function(context, payload, modules, plugin) {
                    // Example custom handler
                    context.response.write(JSON.stringify({
                        plugin: plugin.name,
                        message: 'Query Logger is active'
                    }));
                }
            }
        },

        // Optional: Client-side functionality
        client: {
            // Client-side hooks
            hooks: {
                // Called after app initialization
                onInit: function(data) {
                    console.log('[Query Logger] Plugin initialized');
                    return data;
                },

                // Called after successful query execution
                onAfterQuery: function(data) {
                    var settings = SQT.plugins.loadSettings('query-logger') || { enabled: true };

                    if (settings.enabled) {
                        console.log('[Query Logger] Query executed:', {
                            rowCount: data.rowCount,
                            elapsedTime: data.elapsedTime + 'ms',
                            query: data.query.substring(0, 200) + (data.query.length > 200 ? '...' : '')
                        });

                        // Update status indicator
                        var indicator = document.getElementById('queryLoggerIndicator');
                        if (indicator) {
                            indicator.title = 'Last query: ' + data.rowCount + ' rows in ' + data.elapsedTime + 'ms';
                            indicator.classList.add('sqt-query-logged');
                            setTimeout(function() {
                                indicator.classList.remove('sqt-query-logged');
                            }, 2000);
                        }
                    }

                    return data;
                },

                // Called when editor content changes
                onEditorChange: function(data) {
                    var settings = SQT.plugins.loadSettings('query-logger') || { enabled: true };

                    if (settings.logEditorChanges) {
                        console.log('[Query Logger] Editor changed, length:', data.content.length);
                    }

                    return data;
                }
            },

            // Client-side initialization function
            // Returns an object that becomes the plugin's public API
            init: function(meta, pluginsApi) {
                console.log('[Query Logger] Initializing', meta.name, 'v' + meta.version);

                // Load settings
                var settings = pluginsApi.loadSettings('query-logger') || {
                    enabled: true,
                    logEditorChanges: false
                };

                // Return public API for this plugin
                return {
                    enable: function() {
                        settings.enabled = true;
                        pluginsApi.saveSettings('query-logger', settings);
                        console.log('[Query Logger] Logging enabled');
                    },
                    disable: function() {
                        settings.enabled = false;
                        pluginsApi.saveSettings('query-logger', settings);
                        console.log('[Query Logger] Logging disabled');
                    },
                    isEnabled: function() {
                        return settings.enabled;
                    },
                    getSettings: function() {
                        return settings;
                    }
                };
            }
        },

        // Optional: UI content to inject at designated points
        // Available points: toolbar-start, toolbar-end, export-menu, options-panel,
        // sidebar-section, results-footer, modals, status-bar
        ui: {
            // Inject into status bar
            'status-bar': '<span id="queryLoggerIndicator" class="sqt-plugin-indicator" title="Query Logger active" style="margin-left: 8px; padding: 2px 8px; font-size: 11px; background: var(--sqt-bg-tertiary); border-radius: 4px; cursor: help;"><i class="bi bi-journal-text" style="margin-right: 4px;"></i>Logger</span><style>.sqt-query-logged { background: var(--sqt-accent) !important; color: white !important; transition: background 0.3s; }</style>',

            // Inject into options panel
            'options-panel': '<div class="sqt-options-section" data-sqt-plugin="query-logger"><div class="sqt-options-label">Query Logger</div><div class="sqt-option-row"><input type="checkbox" id="optQueryLoggerEnabled" checked onchange="var p = SQT.plugins.get(\'query-logger\'); if(p) { this.checked ? p.enable() : p.disable(); }"><label for="optQueryLoggerEnabled">Enable console logging</label></div></div>'
        },

        // Optional: Settings schema for automatic UI generation
        // This provides a standardized way to define plugin settings
        settings: {
            schema: {
                enabled: {
                    type: 'boolean',
                    default: true,
                    label: 'Enable Query Logging',
                    description: 'Log query execution details to browser console'
                },
                logEditorChanges: {
                    type: 'boolean',
                    default: false,
                    label: 'Log Editor Changes',
                    description: 'Also log when the query editor content changes'
                }
            }
        }
    };
})()
